﻿using System;
using System.IO;
using System.Text;

namespace dotnet_rings_rust
{
    /// <summary>
    /// Provides methods for reading and writing length-prefixed strings to a stream using Unicode or UTF-8 encoding.
    /// </summary>
    /// <remarks>The StreamString class is designed to facilitate communication over streams by encoding
    /// strings with a two-byte length prefix, followed by the string data. This format enables interoperability between
    /// .NET and other platforms, such as Rust, by supporting both Unicode and UTF-8 encodings. The maximum supported
    /// string length is 65,535 bytes. Instances of StreamString are not thread-safe; concurrent access to the
    /// underlying stream should be synchronized externally.</remarks>
    public class StreamString
    {
        private readonly Stream ioStream;
        private readonly UnicodeEncoding streamEncoding;
        private readonly UTF8Encoding streamEncodingUTF8;

        public StreamString(Stream ioStream)
        {
            this.ioStream = ioStream;
            streamEncoding = new UnicodeEncoding(); // for .NET clients
            streamEncodingUTF8 = new UTF8Encoding(); // for Rust clients
        }

        public string ReadString()
        {
            int len = 0;

            len = ioStream.ReadByte() * 256;
            len += ioStream.ReadByte();

            byte[] inBuffer = new byte[len];
            ioStream.Read(inBuffer, 0, len);

            return streamEncoding.GetString(inBuffer);
        }

        public string ReadUTF8String()
        {
            int len = ioStream.ReadByte() * 256;
            len += ioStream.ReadByte();

            byte[] inBuffer = new byte[len];
            ioStream.Read(inBuffer, 0, len);

            return streamEncodingUTF8.GetString(inBuffer);
        }

        public int WriteString(string outString)
        {
            byte[] outBuffer = streamEncoding.GetBytes(outString);
            int len = outBuffer.Length;

            // The maximum length of a message is 64KB (you can change this limit to suit your needs)
            if (len > UInt16.MaxValue)
            {
                len = (int)UInt16.MaxValue;
            }

            // Write the length in the first two bytes because the len variable is UInt16
            ioStream.WriteByte((byte)(len / 256));
            ioStream.WriteByte((byte)(len & 255));

            // Write the string itself
            ioStream.Write(outBuffer, 0, len);
            ioStream.Flush();

            return outBuffer.Length + 2;
        }

        public int WriteUTF8String(string outString)
        {
            byte[] outBuffer = streamEncodingUTF8.GetBytes(outString);
            int len = outBuffer.Length;

            // The maximum length of a message is 64KB (you can change this limit to suit your needs)
            if (len > UInt16.MaxValue)
            {
                len = (int)UInt16.MaxValue;
            }

            // Write the length in the first two bytes because the len variable is UInt16
            ioStream.WriteByte((byte)(len / 256));
            ioStream.WriteByte((byte)(len & 255));

            // Write the string itself
            ioStream.Write(outBuffer, 0, len);
            ioStream.Flush();

            return outBuffer.Length + 2;
        }
    }
}