use std::io;
use tokio::io::{AsyncReadExt, AsyncWriteExt};
use tokio::net::windows::named_pipe::{ClientOptions, NamedPipeClient};

#[tokio::main]
async fn main() -> io::Result<()> {
    connect_to_pipe().await;
    Ok(())
}

/// Connect to a server listening through a named pipe.
async fn connect_to_pipe() {
    // Define the full path to the named pipe
    let pipe_name = r"\\.\pipe\testpipe";

    // You have to set the impersonation level you want to use with named pipes in Windows.
    // Impersonation levels define what the server process can do on behalf of the client.
    // These are the impersonation levels that can be used:
    // SECURITY_ANONYMOUS=0u32;
    // SECURITY_IDENTIFICATION=65536u32;
    // SECURITY_IMPERSONATION=131072u32
    // SECURITY_DELEGATION=196608u32

    // A rule of thumb for impersonation levels is:
    // Use SECURITY_IMPERSONATION when staying on the same machine;
    // use SECURITY_DELEGATION when the user’s identity must travel across machines.

    const SECURITY_IMPERSONATION: u32 = 131072u32;

    // Attempt to connect to the named pipe
    match ClientOptions::new()
        .security_qos_flags(SECURITY_IMPERSONATION)
        .open(pipe_name)
    {
        Ok(mut pipe) => {
            println!("Successfully connected to the named pipe: {}", pipe_name);

            match read_string(&mut pipe).await {
                Ok(received_string) => {
                    println!("Received string from pipe: {}", received_string);
                }
                Err(e) => {
                    eprintln!("Failed to read from the pipe: {}", e);
                }
            }

            let message = String::from("C:\\DeploymentReport_4.txt");
            if let Err(e) = write_string(message, &mut pipe).await {
                eprintln!("Failed to write to the pipe: {}", e);
            } else {
                println!("Message sent to the pipe.");
            }

            match read_string(&mut pipe).await {
                Ok(received_string) => {
                    println!("Response received string from pipe: {}", received_string);
                }
                Err(e) => {
                    eprintln!("Failed to read from the pipe: {}", e);
                }
            }
        }
        Err(e) => {
            eprintln!("Failed to connect to the named pipe: {}", e);
        }
    }
}

/// Writes a string to the named pipe.
/// The first two bytes of the message have the length of the string.
async fn write_string(
    out_string: String,
    pipe: &mut NamedPipeClient,
) -> Result<i32, std::io::Error> {
    let bytes = out_string.as_bytes();
    let mut len = bytes.len();

    if len > u16::MAX as usize {
        len = u16::MAX as usize;
    }

    let _ = pipe.write_all(&(len as u16).to_be_bytes()).await;
    let _ = pipe.write_all(out_string.as_bytes()).await;

    Ok(bytes.len() as i32 + 2)
}

/// Reads a UTF8 string from the named pipe.
/// The first two bytes of the message have the length of the string.
async fn read_string(pipe: &mut NamedPipeClient) -> Result<String, std::io::Error> {
    let mut len_bytes = [0u8; 2];
    let _ = pipe.read_exact(&mut len_bytes).await;
    let len = u16::from_be_bytes(len_bytes);

    let mut buffer = vec![0; len as usize];
    let _ = pipe.read_exact(&mut buffer).await;

    Ok(String::from_utf8_lossy(&buffer).to_string())
}